#region References

using System;
using System.Text;
using System.Threading;
using System.Data;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.Common.Log;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;
using TABLES = gov.va.med.vbecs.Common.VbecsTables;

#endregion

namespace gov.va.med.vbecs.DAL.HL7AL
{
	#region Header

	//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	//<Developers>
	//	<Developer>Brian Tomlin</Developer>
	//</Developers>
	//<SiteName>Hines OIFO</SiteName>
	//<CreationDate>10/14/2004</CreationDate>
	//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	//<summary></summary>

	#endregion

	/// <summary>
	/// Renamed and modified slightly for BCE (CR 2961)
	/// </summary>
	public class VbecsHL7ClientMonitor
	{
		#region Inner Classes

		#region PendingTransmissionMessage

		/// <summary>
		/// PendingTransmissionMessage
		/// </summary>
		private sealed class PendingTransmissionMessage
		{
			#region Variables

			int _interfaceControlId;
			int _transmitCount;

			string _messageStream;

			#endregion

			#region Properties
			
			/// <summary>
			/// 
			/// </summary>
			internal int InterfaceControlId
			{
				get
				{
					return _interfaceControlId;
				}
				set
				{
					_interfaceControlId = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal string MessageStream
			{
				get
				{
					return _messageStream;
				}
				set
				{
					_messageStream = value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			internal int TransmitCount
			{
				get
				{
					return _transmitCount;
				}
				set
				{
					_transmitCount = value;
				}
			}

			#endregion
		}

		#endregion

		#region HL7ClientMonitor

		/// <summary>
		/// HL7ClientMonitor subclass
		/// </summary>
		public class HL7ClientMonitor
		{
            // Events Logger
            private readonly ILogger _eventsLogger =
                LogManager.Instance().LoggerLocator.GetLogger("SystemEvents");

			/// <summary>
			/// Static Constructor for HL7ClientMonitor
			/// CR 3082: removed caching of interface parameters
			/// </summary>
			static HL7ClientMonitor() {}

			/// <summary>
			/// Constructor for HL7ClientMonitor
			/// </summary>
			public HL7ClientMonitor() {}

		    /// <summary>
		    /// CheckForNewClientMessages
		    /// </summary>
		    /// <param name="state"></param>
		    public void GetAndSendHL7Message(object state)
		    {
		        PendingTransmissionMessage pendingTransmissionMessage = GetHL7MessageFromDb();
				//
				if ( pendingTransmissionMessage != null )
				{
					// CR 3082
					HL7Interface currentInterface = HL7Interface.GetInterfaceById( pendingTransmissionMessage.InterfaceControlId );
					//
					HL7ProtocolMessage newHL7Message = GetHL7Message( pendingTransmissionMessage.MessageStream );
					//
					if ( pendingTransmissionMessage.TransmitCount == currentInterface.ReTransmitAttempts )
					{
                        //CR 3276

                        // Build stripped hl7 message without the full HL7 details (remove PID)
                        var emailMessageBuilder = new StringBuilder();
                        emailMessageBuilder.Append("\n\nThe Message ID is: ");
                        emailMessageBuilder.Append(HL7Utility.ParseGetRequiredMessageSegment(pendingTransmissionMessage.MessageStream, SegmentTypeNames.MSH));
                        emailMessageBuilder.Append("\n\nPlease check the server event log for the full HL7 message details.");
						// If the message has been re-sent up to the maximum number of times, 
						// update the message status, write the event and email the interface administrator
						// CR 2965
						//
                        //CR 3276
                        string message = string.Concat("Maximum number of retry attempts met for message: \n\n\n", emailMessageBuilder.ToString(), "\n HL7 Interface:\n", currentInterface.GetInstanceData(false));
						//
						// Update Message Status
						if ( newHL7Message != null )
						{
							HL7MessageLog.UpdateMessageStatus( 
								MessageStatus.GeneralTransmissionError,
								newHL7Message.GetMessageControlID(),
								message,
								pendingTransmissionMessage.TransmitCount,
								"VBECS HL7 Client Monitor",
								UpdateFunction.VbecsHL7ClientMonitor );
						}
						//
						// Email Interface Administrator
						new HL7MailMessage().SendMessage(currentInterface.InterfaceAdministratorEmail, PII                               ", "Error Sending Message", message, "smtp.DNS   ");
						//
						// Record Event
                        // EventLogAppender
                        _eventsLogger.Warn(string.Format("{0}: {1}", "VBECS HL7 Client Monitor", message));
						//System.Diagnostics.EventLog.WriteEntry( "VBECS HL7 Client Monitor", message, EventLogEntryType.Warning );
					}
					//
					if( newHL7Message != null )
					{
						try
						{
							// CR 2967
							VbecsHL7Bridge.SendVbecsOutboundMessage( newHL7Message, currentInterface, pendingTransmissionMessage.TransmitCount );
						}
						catch(Exception exc)
						{
							// Exception: email the interface administrator
							try
							{
								var mail = new HL7MailMessage();
                                mail.SendMessage(currentInterface.InterfaceAdministratorEmail, PII                               ", "Error Sending Message", HL7Utility.GetInnerExceptionFromException(exc), "smtp.DNS   ");
							}
							catch(Exception emailError)
							{
                                _eventsLogger.Error(string.Format("{0}: {1}", "VBECS HL7 Client Monitor: Failed to send email. Error: ", HL7Utility.GetInnerExceptionFromException(emailError)));
							}
							throw;
						} // catch(Exception exc)
					} // if( newHL7Message != null )
				} // if ( pendingTransmissionMessage != null )
			}

			/// <summary>
			/// 
			/// </summary>
			private static HL7ProtocolMessage GetHL7Message( string newMessageData )
			{
				HL7ProtocolMessage hL7ProtocolMessage = null;
				//
				HL7ProtocolMessageFactory messageFactory = new HL7ProtocolMessageFactory();
				//
				if( newMessageData != string.Empty )
				{
					hL7ProtocolMessage = messageFactory.ParseCreateHL7ProtocolMessage( newMessageData );
				}
				//
				return hL7ProtocolMessage;
			}

			/// <summary>
			/// 
			/// </summary>
			private static PendingTransmissionMessage GetHL7MessageFromDb()
			{
				PendingTransmissionMessage pendingTransmissionMessage = null;
				//
				DataRow drPendingTransmission = Common.StoredProcedure.GetSingleDataRow(Common.VbecsStoredProcs.HL7GetPendingTransmissionMessages.StoredProcName);
				//
				if( drPendingTransmission != null )
				{
					pendingTransmissionMessage = new PendingTransmissionMessage();
					pendingTransmissionMessage.InterfaceControlId = Convert.ToInt32(drPendingTransmission[TABLES.MessageLog.InterfaceControlId]);
					pendingTransmissionMessage.MessageStream = Convert.ToString(drPendingTransmission[TABLES.MessageLog.MessageStream]);
					pendingTransmissionMessage.TransmitCount = Convert.ToInt32(drPendingTransmission[TABLES.MessageLog.TransmitCount]);
				}
				//
				return pendingTransmissionMessage;
			}
		}

		#endregion

		#endregion

		#region Variables

		private Thread _monitorThread;
		private System.Threading.Timer tmrHl7ClientMonitor;

		#endregion

		#region Constructors

		/// <summary>
		/// Static Constructor for VbecsHL7ClientMonitor
		/// </summary>
		static VbecsHL7ClientMonitor() {}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8271"> 
		///		<ExpectedInput>No input required</ExpectedInput>
		///		<ExpectedOutput>Non-null instance of CprsHL7ClientMonitor.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8272"> 
		///		<ExpectedInput>No failure conditions available.</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Blank Constructor for VbecsHL7ClientMonitor
		/// </summary>
		public VbecsHL7ClientMonitor() {}

		#endregion

		#region Public Methods

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8273"> 
		///		<ExpectedInput>This method is not testable. The underlying class HL7ClientMonitor is tested.</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8274"> 
		///		<ExpectedInput>This method is not testable. The underlying class HL7ClientMonitor is tested.</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Start
		/// </summary>
		public void Start()
		{
			lock( this )
			{
				_monitorThread = new Thread( new ThreadStart( TimerStart ) );
				_monitorThread.Start();
			}
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8275"> 
		///		<ExpectedInput>This method is not testable.</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8276"> 
		///		<ExpectedInput>This method is not testable. The underlying class HL7ClientMonitor is tested.</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Stop
		/// </summary>
		public void Stop()
		{
			lock( this )
			{
				if( _monitorThread == null )
				{
					return;
				}
				//
				_monitorThread.Abort();
				//
				_monitorThread.Join( 5000 );
				//
				tmrHl7ClientMonitor.Change(Timeout.Infinite, Timeout.Infinite );
			}
		}

		#endregion

		#region Private Methods

		/// <summary>
		/// 
		/// </summary>
		private void TimerStart()
		{
			lock( this )
			{
				tmrHl7ClientMonitor = new Timer( new TimerCallback( new HL7ClientMonitor().GetAndSendHL7Message ) , null, 0, 5000 );
			}
		}

		#endregion
	}
}
